%%%% A Matlab Code for 2D Non-Penalization SEMDOT Algorithm %%%%
% Example SEMDOT2D(100,100,0.3,3)
function SEMDOT2D(nelx,nely,volfrac,rmin)
%% INITIALIZATION
nodenrs=reshape(1:(1+nelx)*(1+nely),1+nely,1+nelx);
edofVec=reshape(nodenrs(1:end-1,1:end-1)+1,nelx*nely,1);
edofMat=repmat(edofVec,1,4)+repmat([0 nely+[1 0] -1],nelx*nely,1);
iK=reshape(kron(edofMat,ones(4,1))',16*nelx*nely,1);
jK=reshape(kron(edofMat,ones(1,4))',16*nelx*nely,1);
%% GEOMETRY & MATERIAL
E0=10;   Emin=0.1; M0=1e6;  Mmin=5e5; dx=0.1/nelx; dy=dx; h=1e-3;
[KE,ME]=Elementmatrix(dx,dy,h);
%% FILTER
[H,Hs]=HHs(nelx,nely,rmin);
[Hn,Hns]=HnHns(nelx,nely);
%% OPTIMIZATION
x=repmat(volfrac,nely,nelx); xPhys=x; nG=10; ngrid=nG-1;
%% DEFINE SUPPORTS
fixeddofs=[1,nely+1,nelx*(nely+1)+1,(nelx+1)*(nely+1)];
alldofs=[1:(nelx+1)*(nely+1)]; freedofs=setdiff(alldofs,fixeddofs);
%% TRANSIENT PARA
totalt=1000;
nn=500; dett=totalt/nn; theta=1;
%% DEFINE LOADS
F=sparse((nelx+1)*(nely+1),nn+1);
Amplitude=1; ratio=0.5;
F(((nelx+1)*(nely+1)+1)/2,1:ratio*nn+1)=Amplitude;
%% PROJECTION
beta=0.5; ER=0.5;
%% MMA
m=1; n=nelx*nely; xold1=x(:); xold2=xold1;
xmin=zeros(n,1); xmax=ones(n,1); low=xmin; upp=xmax;
a0=1; a=zeros(m,1); cmma=10000*ones(m,1); d=zeros(m,1);
move=0.1; scale_obj=1; scale_cons=1000;
%% ELEMENTAL NODES AND COORDINATES
[nodex,nodey]=meshgrid(0:nelx,0:nely);
[fnx,fny]=meshgrid(0:1/ngrid:nelx,0:1/ngrid:nely);
%% CONVERGENCE
tolx=0.001; maxloop=5000; iter=0; change=1;
%% START ITERATION
while (change > tolx || tol>0.001) && iter < maxloop
    iter=iter+1;
    beta=beta+ER;
    Con(iter,1)=sum(xPhys(:))/nelx/nely;
    %% FEA
    sK=reshape(KE(:)*(E0*xPhys(:)'+Emin*(1-xPhys(:)')),16*nelx*nely,1);
    sM=reshape(ME(:)*(M0*xPhys(:)'+Mmin*(1-xPhys(:)')),16*nelx*nely,1);
    K=sparse(iK,jK,sK); K=(K+K')/2;
    M=sparse(iK,jK,sM); M=(M+M')/2;
    KM=M/dett+theta*K;  dKM=decomposition(KM(freedofs,freedofs));
    KM1=M/dett-(1-theta)*K;
    %% SOVLE
    T=zeros((nelx+1)*(nely+1),nn+1); c=0;
    for it=2:nn+1
        Feffc=KM1*T(:,it-1)+(1-theta)*F(:,it-1)+theta*F(:,it);
        T(freedofs,it)=dKM\Feffc(freedofs,1);
        c=c+T(:,it)'*K*T(:,it)*dett/totalt;
    end
    %% ADJOINT
    for it=1:nn+1, P(:,it)=-2*K*T(:,nn+2-it); end;
    Ta=zeros((nelx+1)*(nely+1),nn+1);
    for it=2:nn+1
        Feffc=KM1*Ta(:,it-1)+(1-theta)*P(:,it-1)+theta*P(:,it);
        Ta(freedofs,it)=dKM\Feffc(freedofs,1);
    end
    for it=1:nn+1, Tb(:,nn+2-it)=Ta(:,it); end
    %% SENSIVITIY
    dc=zeros(nely,nelx);
    for it=2:nn+1
        U1=T(:,it); U2=Tb(:,it);
        if (it==2) | (it==nn+1), V1=(T(:,it)-T(:,it-1))/dett; else, V1=(T(:,it+1)-T(:,it-1))/dett/2; end
        ce1=reshape(sum((U2(edofMat)*ME).*V1(edofMat),2),nely,nelx);
        ce2=reshape(sum((U2(edofMat)*KE).*U1(edofMat),2),nely,nelx);
        dc=dc+(E0*xPhys+Emin*(1-xPhys)).*(ce1+ce2);
    end
    dc=dc*dett/totalt;
    dm=ones(nely,nelx)/nelx/nely;
    Obj(iter,1)=c;
    %% FILTERING
    dc(:)=H*(dc(:)./Hs);
    dm(:)=H*(dm(:)./Hs);
    %% MMA OBJ % CONS
    f0val=scale_obj*c;
    df0dx=scale_obj*dc(:);
    fval(1,1)=scale_cons*(Con(end)/volfrac-1);
    dfdx(1,:)=scale_cons*dm(:)'/volfrac;
    %% MMA SOLVER
    dv=x(:); xmax=min(1,dv+move); xmin=max(0,dv-move);
    [xmma,~,~,~,~,~,~,~,~,low,upp]=...
        mmasub(m,n,iter,dv,xmin,xmax,xold1,xold2,f0val,df0dx,fval,dfdx,low,upp,a0,a,cmma,d);
    if iter>2, xold2=xold1; xold1=dv; elseif iter>1, xold1=dv; end
    xnew=reshape(xmma,nely,nelx); xPhys(:)=(H*xnew(:))./Hs;
    %% UPDATE POINT DESNIGY BY A HEAVISIDE SMOOTH/STEP FUNCTION
    xg=interp_xg(nelx,nely,xPhys,Hn,Hns,nodex,nodey,fnx,fny);
    [ls,xPhys,Terr]=updatexPhys(nelx,nely,xg,ngrid,xPhys,beta);
    change=sum(abs(xnew(:)-x(:)))/(volfrac*nely*nelx);
    tol=Terr/(nely*nelx);
    x=xnew;
    %% PLOT RESULTS
    fprintf('It.:%3i Obj.:%6.3f Vol.:%4.3f ch.:%4.5f Topo.:%7.5f beta.:%g.\n',iter,Obj(end),Con(end),change,tol,beta);
    contourf(fnx, flipud(fny), xg-ls, [0 0]); axis equal tight off; drawnow
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This code was written by:                                               %
%                                                                         %
% Dr. Yun-Fei Fu                                                          %
% Department of Mechanical Engineering, University of Alberta, Canada     %
%                                                                         %
% Prof. Bernard Rolfe                                                     %
% School of Engineering, Faculty of Science Engineering and               %
% Built Environment, Deakin University, Australia                         %
%                                                                         %
% Please send your comments and questions to: jackyunfei.fu@outlook.com   %
%                                                                         %
% The code is intended for educational purposes and theoretical details   %
% are discussed in the book: "Smooth Topological Design of Continuum      %
% Structures", YF Fu and B Rolfe, CRC press.                              %
%                                                                         %
%                                                                         %
% Disclaimer:                                                             %
% The authors do not guarantee that the code is free from errors.         %
% Furthermore, the authors shall not be liable in any event caused by the %
% use of the program.                                                     %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%